/*!
 * @file        main.c
 *
 * @brief       Main program body
 *
 * @version     V1.0.2
 *
 * @date        2022-06-23
 *
 * @attention
 *
 *  Copyright (C) 2021-2022 Geehy Semiconductor
 *
 *  You may not use this file except in compliance with the
 *  GEEHY COPYRIGHT NOTICE (GEEHY SOFTWARE PACKAGE LICENSE).
 *
 *  The program is only for reference, which is distributed in the hope
 *  that it will be usefull and instructional for customers to develop
 *  their software. Unless required by applicable law or agreed to in
 *  writing, the program is distributed on an "AS IS" BASIS, WITHOUT
 *  ANY WARRANTY OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the GEEHY SOFTWARE PACKAGE LICENSE for the governing permissions
 *  and limitations under the License.
 */

/* Includes */
#include "main.h"
#include "Board.h"
#include "stdio.h"
#include "apm32f4xx_iwdt.h"

/** @addtogroup Examples
  @{
  */

/** @addtogroup IWDT_Reset
  @{
  */

/** @defgroup IWDT_Reset_Macros Macros
  @{
*/

/* printf using USART1  */
#define DEBUG_USART  USART1

/**@} end of group IWDT_Reset_Macros*/

/** @defgroup IWDT_Reset_Variables Variables
  @{
  */

volatile uint32_t TimingDelay = 0;
volatile uint8_t bKEY1_Flag = 0;

/**@} end of group IWDT_Reset_Variables*/

/** @defgroup IWDT_Reset_Functions Functions
  @{
  */

/* USART Initialization */
void USART_Init(void);
void Delay(void);

/*!
 * @brief     Main program
 *
 * @param     None
 *
 * @retval    None
 */
int main(void)
{
    APM_MINI_LEDInit(LED2);
    APM_MINI_LEDInit(LED3);
    APM_MINI_PBInit(BUTTON_KEY1, BUTTON_MODE_EINT);

    /* USART Initialization */
    USART_Init();
    printf("IWDT is running, if press KEY1 to start feed watchdog, Ohterwise System will reset about 4 seconds!\r\n");

    SysTick_Config(SystemCoreClock / 1000);

    if(RCM_ReadStatusFlag(RCM_FLAG_IWDTRST) == SET)
    {
        /* IWDT Reset */
        APM_MINI_LEDOn(LED3);
        printf("\r\n IWDT Reset! \r\n");
        RCM_ClearStatusFlag();
    }
    else
    {
        APM_MINI_LEDOff(LED3);
    }

    IWDT_EnableWriteAccess();

    /* Colck is 1000Hz(32k/32) */
    IWDT_ConfigDivider(IWDT_DIVIDER_32);

    /* About 4 seconds */
    IWDT_ConfigReload(4000);
    IWDT_Enable();

    while(1)
    {
        if(bKEY1_Flag == 1)
        {
            APM_MINI_LEDToggle(LED2);

            Delay();

            IWDT_Refresh();
            printf("\r\nif press KEY1 again to finish feeding watchdog, System will reset about 4 seconds!\r\n");
        }
        else
        {
            APM_MINI_LEDOff(LED2);
        }
    }
}

/*!
 * @brief     USART Initialization
 *
 * @param     None
 *
 * @retval    None
 */
void USART_Init(void)
{
    USART_Config_T usartConfigStruct;

    /* USART configuration */
    USART_ConfigStructInit(&usartConfigStruct);
    usartConfigStruct.baudRate = 115200;
    usartConfigStruct.mode = USART_MODE_TX_RX;
    usartConfigStruct.parity = USART_PARITY_NONE;
    usartConfigStruct.stopBits = USART_STOP_BIT_1;
    usartConfigStruct.wordLength = USART_WORD_LEN_8B;
    usartConfigStruct.hardwareFlow = USART_HARDWARE_FLOW_NONE;

    /* COM1 init*/
    APM_MINI_COMInit(COM1, &usartConfigStruct);
}

/*!
 * @brief     Delay
 *
 * @param     None
 *
 * @retval    None
 */
void Delay(void)
{
    TimingDelay = 0;
    while(TimingDelay < 300);
}

/*!
 * @brief     Redirect C Library function printf to serial port.
 *            After Redirection, you can use printf function.
 *
 * @param     ch:  The characters that need to be send.
 *
 * @param     *f:  pointer to a FILE that can recording all information
 *            needed to control a stream
 *
 * @retval    The characters that need to be send.
 */
int fputc(int ch, FILE* f)
{
    /* send a byte of data to the serial port */
    USART_TxData(DEBUG_USART, (uint8_t)ch);

    /* wait for the data to be send  */
    while (USART_ReadStatusFlag(DEBUG_USART, USART_FLAG_TXBE) == RESET);

    return (ch);
}

/**@} end of group IWDT_Reset_Functions */
/**@} end of group IWDT_Reset */
/**@} end of group Examples */
